<?php

namespace App\Http\Controllers;

use App\Models\Journal;
use App\Models\JournalCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class JournalCategoryController extends Controller {
    public function index(Request $request) {

        if ($request->ajax()) {
            $search = $request->get('search');

            $query = JournalCategory::query();

            if ($search) {
                $query->whereAny([
                    'name',
                ], 'like', "%{$search}%")
                    ->orWhereHas('journal', function ($q) use ($search) {
                        $q->whereAny([
                            'name',
                        ], 'like', "%{$search}%");
                    });
            }

            $data      = $query->paginate()->withQueryString();
            $rowsHtml  = view('journal-category.index-data-rows', compact('data'))->render();
            $linksHtml = $data->links('vendor.pagination.custom-pagination')->render();

            return response()->json([
                'rows'  => $rowsHtml,
                'links' => $linksHtml,
            ]);
        }

        $data = JournalCategory::paginate();

        return view('journal-category.index', compact('data'));
    }

    public function create() {
        $journals = Journal::orderBy('name')->get();

        return view('journal-category.create', compact('journals'));
    }

    public function store(Request $request) {
        $validator = Validator::make($request->all(), [
            'name'       => ['required', 'max:255', 'unique:journal_categories,name'],
            'journal_id' => ['required'],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {

            JournalCategory::create([
                'name'       => $request->name,
                'slug'       => Str::slug($request->name),
                'journal_id' => $request->journal_id,
            ]);

            return storeResponse();
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function edit($id) {
        $data = JournalCategory::find($id);

        if (!$data) {
            return errorResponse();
        }

        $journals = Journal::orderBy('name')->get();

        return view('journal-category.edit', compact('data', 'journals'));
    }

    public function update(Request $request, $id) {
        $validator = Validator::make($request->all(), [
            'name'       => ['required', 'string', 'max:255', Rule::unique('journal_categories', 'name')->ignore($id)],
            'journal_id' => ['required'],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {
            $data = JournalCategory::find($id);

            if (!$data) {
                return errorResponse();
            }

            $payload = [
                'name'       => $request->name,
                'slug'       => Str::slug($request->name),
                'journal_id' => $request->journal_id,
            ];

            $data->update($payload);

            return updateResponse('journal-categories.index');
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function destroy($id) {
        $data = JournalCategory::find($id);

        if (!$data) {
            return errorResponse();
        }

        $data->delete();

        return deleteResponse();
    }

}
