<?php

namespace App\Http\Controllers;

use App\Models\Journal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class JournalController extends Controller 
{
    public function index(Request $request) {

        if ($request->ajax()) {
            $search = $request->get('search');

            $query = Journal::query();

            if ($search) {
                $query->whereAny([
                    'name',
                    'issn_digital',
                    'coden',
                    'launched_year',
                ], 'like', "%{$search}%");
            }

            $data      = $query->paginate()->withQueryString();
            $rowsHtml  = view('journal.index-data-rows', compact('data'))->render();
            $linksHtml = $data->links('vendor.pagination.custom-pagination')->render();

            return response()->json([
                'rows'  => $rowsHtml,
                'links' => $linksHtml,
            ]);
        }

        $data = Journal::paginate();

        return view('journal.index', compact('data'));
    }

    public function create() {

        return view('journal.create');
    }

    public function store(Request $request) {
        $validator = Validator::make($request->all(), [
            'name'          => ['required', 'max:255', 'unique:journals,name'],
            'abr_name'      => ['required'],
            'launched_year' => ['required', 'integer'],
            'j_format'      => ['nullable', 'mimes:docx'],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {
            $j_format = null;

            if ($request->hasFile('j_format')) {
                $file     = $request->file('j_format');
                $j_format = uploadImage('journal_format', $file);
            }

            $icon = null;

            if ($request->hasFile('icon')) {
                $file = $request->file('icon');
                $icon = uploadImage('journal_icon', $file);
            }

            Journal::create([
                'name'                  => $request->name,
                'slug'                  => Str::slug($request->name),
                'abr_name'              => $request->abr_name,
                'issn_print'            => $request->issn_print,
                'issn_digital'          => $request->issn_digital,
                'coden'                 => $request->coden,
                'publication_frequency' => $request->publication_frequency,
                'publication_medium'    => $request->publication_medium,
                'launched_year'         => $request->launched_year,
                'icon'                  => $icon,
                'color_family'          => $request->color_family,
                'description'           => $request->description,
                'rating'                => $request->rating,
                'j_format'              => $j_format,
            ]);

            return storeResponse();
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function show($id) {
        $data = Journal::where('id', $id)
            ->with(
                'journalCategories',
                'editors',
                'reviewers',
                'assignments.assignedBy'
            )
            ->first();

        if (!$data) {
            return errorResponse();
        }

        return view('journal.show', compact('data'));
    }

    public function edit($id) {
        $data = Journal::find($id);

        if (!$data) {
            return errorResponse();
        }

        return view('journal.edit', compact('data'));
    }

    public function update(Request $request, $id) {
        $validator = Validator::make($request->all(), [
            'name'          => ['required', 'string', 'max:255', Rule::unique('journals', 'name')->ignore($id)],
            'abr_name'      => ['required'],
            'launched_year' => ['required', 'integer'],
            'j_format'      => ['nullable', 'mimes:docx'],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {
            $data = Journal::find($id);

            if (!$data) {
                return errorResponse();
            }

            if ($request->hasFile('j_format')) {
                $file = $request->file('j_format');

                if ($data->j_format) {
                    $image_path = public_path($data->j_format);

                    if (File::exists($image_path)) {
                        File::delete($image_path);
                    }

                }

                $newImage = uploadImage('journal_format', $file);
                $data->update(['j_format' => $newImage]);
            }

            if ($request->hasFile('icon')) {
                $file = $request->file('icon');

                if ($data->icon) {
                    $image_path = public_path($data->icon);

                    if (File::exists($image_path)) {
                        File::delete($image_path);
                    }

                }

                $newImage = uploadImage('journal_icon', $file);
                $data->update(['icon' => $newImage]);
            }

            $payload = [
                'name'                  => $request->name,
                'slug'                  => Str::slug($request->name),
                'abr_name'              => $request->abr_name,
                'issn_print'            => $request->issn_print,
                'issn_digital'          => $request->issn_digital,
                'coden'                 => $request->coden,
                'publication_frequency' => $request->publication_frequency,
                'publication_medium'    => $request->publication_medium,
                'launched_year'         => $request->launched_year,
                'color_family'          => $request->color_family,
                'description'           => $request->description,
                'rating'                => $request->rating,
            ];

            $data->update($payload);

            return updateResponse('journals.index');
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function destroy($id) {
        $data = Journal::find($id);

        if (!$data) {
            return errorResponse();
        }

        if ($data->j_format) {
            $image_path = public_path($data->j_format);

            if (File::exists($image_path)) {
                File::delete($image_path);
            }

        }

        if ($data->icon) {
            $image_path = public_path($data->icon);

            if (File::exists($image_path)) {
                File::delete($image_path);
            }

        }

        $data->delete();

        return deleteResponse();
    }

}
