<?php

namespace App\Http\Controllers;

use App\Models\Journal;
use App\Models\JournalUserAssignment;
use App\Models\User;
use Illuminate\Http\Request;

class JournalUserAssignmentController extends Controller
{
    public function create()
    {
        return view('journal-assignments.create-or-update', [
            'journals' => Journal::orderBy('name')->get(),
        ]);
    }

    public function getUsers(Request $request)
    {
        $request->validate([
            'journal_id' => 'required|exists:journals,id',
            'role'       => 'required|in:Reviewer,Editor',
        ]);

        $users = User::where('role', $request->role)->get();

        $assignedUserIds = JournalUserAssignment::where('journal_id', $request->journal_id)
            ->where('role', $request->role)
            ->pluck('assignable_id')
            ->toArray();

        return response()->json([
            'users' => $users,
            'assigned' => $assignedUserIds,
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'journal_id' => 'required|exists:journals,id',
            'role'       => 'required|in:Reviewer,Editor',
            'users'      => 'array',
            'users.*'    => 'exists:users,id',
        ]);

        $journalId = $request->journal_id;
        $role      = $request->role;
        $userIds   = $request->users ?? [];

        // Remove unselected users
        JournalUserAssignment::where('journal_id', $journalId)
            ->where('role', $role)
            ->whereNotIn('assignable_id', $userIds)
            ->delete();

        // Insert selected users (ignore duplicates)
        foreach ($userIds as $userId) {
            JournalUserAssignment::firstOrCreate([
                'journal_id'      => $journalId,
                'assignable_id'   => $userId,
                'assignable_type' => User::class,
                'role'            => $role,
            ], [
                'assigned_by' => auth()->id(),
            ]);
        }

        return redirect()
            ->back()
            ->with('success', ucfirst($role) . ' assigned successfully.');
    }
}
