<?php

namespace App\Http\Controllers;

use App\Models\SiteDesignation;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller {
    public function index(Request $request) {

        if ($request->ajax()) {
            $search = $request->get('search');

            $query = User::query();

            if ($search) {
                $query->whereAny([
                    'name',
                    'email',
                    'role',
                    'bio',
                    'working_history',
                    'sci_profile',
                    'scilit',
                    'preprint',
                    'google_scholar',
                    'hobbies',
                    'gender',
                ], 'like', "%{$search}%");
            }

            $data      = $query->paginate()->withQueryString();
            $rowsHtml  = view('user.index-data-rows', compact('data'))->render();
            $linksHtml = $data->links('vendor.pagination.custom-pagination')->render();

            return response()->json([
                'rows'  => $rowsHtml,
                'links' => $linksHtml,
            ]);
        }

        $data = User::paginate();

        return view('user.index', compact('data'));
    }

    public function create() {
        $designations = SiteDesignation::orderBy('name')->get();

        return view('user.create', compact('designations'));
    }

    public function store(Request $request) {
        $validator = Validator::make($request->all(), [
            'name'      => ['required', 'string', 'max:255'],
            'email'     => ['required', 'email', 'max:255', 'unique:users,email'],
            'password'  => ['required', 'string', 'min:3'],
            'role'      => ['required', Rule::in(['Admin', 'Editor', 'Author'])],
            'is_active' => ['boolean'],
            'image'     => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif,svg', 'max:2048'],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {
            $image = null;

            if ($request->hasFile('image')) {
                $file  = $request->file('image');
                $image = uploadImage('user', $file);
            }

            User::create([
                'name'                => $request->name,
                'gender'              => $request->gender,
                'dob'                 => $request->dob,
                'email'               => $request->email,
                'password'            => bcrypt($request->password),
                'role'                => $request->role,
                'is_active'           => $request->boolean('is_active', true),
                'site_designation_id' => $request->site_designation_id,
                'website'             => $request->website,
                'bio'                 => $request->bio,
                'working_history'     => $request->working_history,
                'sci_profile'         => $request->sci_profile,
                'scilit'              => $request->scilit,
                'preprint'            => $request->preprint,
                'google_scholar'      => $request->google_scholar,
                'hobbies'             => $request->hobbies,
                'address'             => $request->address,
                'image'               => $image,
            ]);

            return storeResponse();
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function edit($id) {
        $data = User::find($id);

        if (!$data || !$data->id == 1) {
            return errorResponse();
        }

        $designations = SiteDesignation::orderBy('name')->get();

        return view('user.edit', compact('data', 'designations'));
    }

    public function update(Request $request, $id) {
        $validator = Validator::make($request->all(), [
            'name'      => ['required', 'string', 'max:255'],
            'email'     => ['required', 'string', 'max:255', Rule::unique('users', 'email')->ignore($id)],
            'password'  => ['nullable', 'string', 'min:3'],
            'role'      => ['required', Rule::in(['Admin', 'Editor', 'Author'])],
            'is_active' => ['boolean'],
            'image'     => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif,svg', 'max:2048'],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {
            $data = User::find($id);

            if (!$data || !$data->id == 1) {
                return errorResponse();
            }

            if ($request->hasFile('image')) {
                $file = $request->file('image');

                if ($data->image) {
                    $image_path = public_path($data->image);

                    if (File::exists($image_path)) {
                        File::delete($image_path);
                    }

                }

                $newImage = uploadImage('user', $file);
                $data->update(['image' => $newImage]);
            }

            $payload = [
                'name'                => $request->name,
                'gender'              => $request->gender,
                'dob'                 => $request->dob,
                'email'               => $request->email,
                'role'                => $request->role,
                'is_active'           => $request->boolean('is_active', true),
                'site_designation_id' => $request->site_designation_id,
                'website'             => $request->website,
                'bio'                 => $request->bio,
                'working_history'     => $request->working_history,
                'sci_profile'         => $request->sci_profile,
                'scilit'              => $request->scilit,
                'preprint'            => $request->preprint,
                'google_scholar'      => $request->google_scholar,
                'hobbies'             => $request->hobbies,
                'address'             => $request->address,
            ];

            if ($request->filled('password')) {
                $payload['password'] = bcrypt($request->password);
            }

            $data->update($payload);

            return updateResponse('users.index');
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function destroy($id) {
        $data = User::find($id);

        if (!$data || !$data->id == 1) {
            return errorResponse();
        }

        if ($data->image) {
            $image_path = public_path($data->image);

            if (File::exists($image_path)) {
                File::delete($image_path);
            }

        }

        $data->delete();

        return deleteResponse();
    }

}
